import React from "react";
import { X, ChevronRight, LogOut } from "lucide-react";
import { useNavigate } from "react-router-dom";
import { useAuth } from "@/context/AuthContext";
import { toast } from "react-hot-toast";
import { useEffect, useState } from "react";

interface AccountModalProps {
  isOpen: boolean;
  onClose: () => void;
}

const AccountModal: React.FC<AccountModalProps> = ({ isOpen, onClose }) => {
  const navigate = useNavigate();
  const { logout, user } = useAuth();
  const [formattedPhone, setFormattedPhone] = useState<string>('');

  useEffect(() => {
    if (user?.phone) {
      // Format phone number for display: +91 XXXX XXX XXX
      const cleaned = user.phone.replace(/\D/g, '');
      const match = cleaned.match(/^(\d{3})(\d{3})(\d{4})$/);
      if (match) {
        setFormattedPhone(`+91 ${match[1]} ${match[2]} ${match[3]}`);
      } else {
        setFormattedPhone(`+91 ${user.phone}`);
      }
    }
  }, [user]);
  
  const handleLogout = async () => {
    try {
      await logout();
      toast.success("Logged out successfully");
      navigate("/login");
      onClose();
    } catch (error) {
      console.error("Logout failed:", error);
      toast.error("Failed to logout. Please try again.");
    }
  };
  
  const handleMenuItemClick = (label: string) => {
    switch (label) {
      case "Logout":
        handleLogout();
        break;
      case "My Order":
        navigate("/home/orders");
        onClose();
        break;
      case "My Address":
        navigate("/home/addresses");
        onClose();
        break;
      case "Voucher":
        navigate("/home/profile");
        onClose();
        break;
      case "Need Help?":
        // Handle help navigation or modal
        break;
      default:
        break;
    }
  };
  if (!isOpen) return null;

  const menuItems = [
    { label: "My Order", icon: null },
    { label: "My Address", icon: null },
    { label: "Need Help?", icon: null },
    { label: "Voucher", icon: null },
    { 
      label: "Logout", 
      icon: <LogOut className="w-4 h-4 text-red-600" />,
      className: "text-red-600 hover:bg-red-50"
    },
  ];

  return (
    <div 
      className={`fixed inset-0 z-[1000] flex justify-end transition-opacity duration-300 ${
        isOpen ? 'opacity-100' : 'opacity-0 pointer-events-none'
      }`}
    >
      {/* Backdrop */}
      <div 
        className="absolute inset-0 bg-black/30 transition-opacity duration-300"
        onClick={onClose}
        aria-hidden="true"
      />
      
      {/* Modal panel */}
      <div 
        className={`relative w-full max-w-sm sm:max-w-xs bg-white h-full shadow-xl flex flex-col z-[1001] transform transition-transform duration-300 ease-in-out ${
          isOpen ? 'translate-x-0' : 'translate-x-full'
        }`}
      >
        {/* Header */}
        <div className="flex items-center justify-between p-4 border-b border-gray-200 sticky top-0 bg-white z-10">
          <h2 className="text-gray-900 font-semibold text-lg sm:text-xl">My Account</h2>
          <button
            onClick={onClose}
            className="p-1 rounded-full hover:bg-gray-100 text-gray-500 hover:text-gray-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2"
            aria-label="Close account menu"
          >
            <X className="w-5 h-5 sm:w-6 sm:h-6" />
          </button>
        </div>

        {/* Content */}
        <div className="flex-1 overflow-y-auto p-4 sm:p-6">
          {/* Phone number */}
          <div className="mb-6 p-3 bg-blue-50 rounded-lg">
            <p className="text-xs text-gray-500">Logged in as</p>
            {user?.phone ? (
              <p className="text-sm sm:text-base font-medium text-gray-900">
                {formattedPhone}
              </p>
            ) : (
              <div className="h-5 bg-gray-200 rounded w-3/4 animate-pulse"></div>
            )}
          </div>

          {/* Menu items */}
          <ul className="space-y-1">
            {menuItems.map((item) => (
              <li key={item.label}>
                <button
                  onClick={() => handleMenuItemClick(item.label)}
                  className={`w-full flex items-center justify-between p-3 sm:p-4 rounded-lg text-left transition-colors ${
                    item.className || 'text-gray-800 hover:bg-gray-50 hover:text-blue-600'
                  }`}
                >
                  <span className="text-sm sm:text-base font-medium">{item.label}</span>
                  <div className="flex items-center">
                    {item.icon || <ChevronRight className="w-4 h-4 sm:w-5 sm:h-5 text-gray-400" />}
                  </div>
                </button>
              </li>
            ))}
          </ul>
        </div>

        {/* Footer (buttons) */}
        <div className="sticky bottom-0 bg-white border-t border-gray-200 p-4">
          <p className="text-xs text-gray-500 text-center mb-3">Get the best experience with our app</p>
          <div className="flex flex-col sm:flex-row gap-3">
            <button 
              className="flex-1 flex items-center justify-center gap-2 bg-blue-600 hover:bg-blue-700 text-white py-2.5 px-4 rounded-lg text-sm font-medium transition-colors"
              onClick={() => {}}
            >
              <svg className="w-4 h-4" viewBox="0 0 24 24" fill="currentColor">
                <path d="M3.609 1.814L13.792 12 3.61 22.186a.996.996 0 0 1-.61-.92V2.734a1 1 0 0 1 .609-.92zm.921 5.095v10.18l7.16-5.09-7.16-5.09zM14.5 12l7.46 7.68c.3.31.04.82-.4.82H17v-3.34l-2.5 1.67V19.5h-1V12h1v3.65l5-3.33V4h3.56c.44 0 .7.51.4.82L14.5 12z"/>
              </svg>
              Play Store
            </button>
            <button 
              className="flex-1 flex items-center justify-center gap-2 bg-gray-900 hover:bg-gray-800 text-white py-2.5 px-4 rounded-lg text-sm font-medium transition-colors"
              onClick={() => {}}
            >
              <svg className="w-4 h-4" viewBox="0 0 24 24" fill="currentColor">
                <path d="M17.05 20.28c-.98.95-2.05.8-3.08.35-1.09-.46-2.09-.48-3.24 0-1.44.62-2.2.44-3.06-.35C2.79 15.25 3.51 7.59 9.05 7.31c1.35.39 2.2 1.32 3.04 1.34.77.02 1.51-.76 2.91-.9 1.2-.15 2.4.22 3.43.83 1.15.71 1.46 1.27 2.29 2.26-2.02 1.19-2.96 2.93-2.77 5.6.19 2.72 1.68 3.82 3 4.54zM12.03 7.25c-.15-2.23 1.66-4.07 3.74-4.25.29 2.58-2.34 4.5-3.74 4.25z"/>
              </svg>
              App Store
            </button>
          </div>
        </div>
      </div>
    </div>
  );
};

export default AccountModal;
