import { Search, Upload, ShieldCheck, Clock, Star, ChevronLeft, ChevronRight } from 'lucide-react';
import { useEffect, useState } from 'react';
import { getActiveBanners } from '@/services/bannerService';
import { Banner } from '@/types/banner';

// Helper function to format banner image URL
const formatBannerImageUrl = (imagePath: string): string => {
  if (!imagePath) return '';
  
  // If it's already a full URL, return as is
  if (imagePath.startsWith('http')) {
    return imagePath;
  }
  
  // Remove any leading slashes or backslashes
  const cleanPath = imagePath.replace(/^[\\/]+/, '');
  
  // Construct the full URL
  const baseUrl = import.meta.env.VITE_API_URL || 'http://localhost:5000';
  return `${baseUrl}/uploads/${cleanPath}`.replace(/([^:]\/)\/+/g, '$1');
};

import { useNavigate } from 'react-router-dom';

// Default hero content when no banners are available
function DefaultHeroContent() {
  const navigate = useNavigate();

  return (
    <div className="relative bg-gradient-to-r from-cyan-200 to-blue-400 rounded-2xl overflow-hidden my-8 mx-auto max-w-7xl p-8">
      <div className="container mx-auto px-6 py-16 text-center text-gray-800 relative z-10">
        <h1 className="text-4xl md:text-5xl font-bold leading-tight text-blue-900">
          India's Most Trusted
          <br />
          Healthcare Platform
        </h1>
        <p className="mt-4 text-base md:text-lg text-gray-600 max-w-2xl mx-auto">
          Order medicines with prescription, consult doctors online - all from the comfort of your home
        </p>
        
        <div className="mt-8 max-w-xl mx-auto">
          <div className="flex items-center bg-white rounded-full p-1.5 shadow-md">
            <Search className="text-gray-400 mx-3 h-5 w-5" />
            <input
              type="text"
              placeholder="Search your needs"
              className="w-full bg-transparent focus:outline-none text-gray-700 placeholder-gray-500"
            />
            <button className="bg-blue-600 hover:bg-blue-700 text-white font-semibold py-2.5 px-8 rounded-full text-sm">
              Search
            </button>
          </div>
        </div>

        <div className="my-4 text-gray-500 text-sm">or</div>

        <button
          onClick={() => navigate('/home/upload-prescription')}
          className="bg-white text-blue-600 font-bold py-3 px-6 rounded-full shadow-md hover:bg-gray-100 flex items-center mx-auto border border-gray-200"
        >
          <Upload className="mr-2 h-5 w-5" />
          <span className="text-sm">Upload Prescription</span>
        </button>

        <div className="mt-8 flex flex-wrap justify-center items-center gap-x-8 gap-y-4 text-gray-600">
          <div className="flex items-center text-sm font-medium">
            <ShieldCheck className="mr-2 h-5 w-5 text-blue-500" />
            <span>100% Authentic</span>
          </div>
          <div className="flex items-center text-sm font-medium">
            <Clock className="mr-2 h-5 w-5 text-blue-500" />
            <span>2 Hour Delivery</span>
          </div>
          <div className="flex items-center text-sm font-medium">
            <Star className="mr-2 h-5 w-5 text-blue-500" />
            <span>4.8 Rating</span>
          </div>
        </div>
      </div>
    </div>
  );
}

export default function Hero() {
  const navigate = useNavigate();
  const [banners, setBanners] = useState<Banner[]>([]);
  const [currentBannerIndex, setCurrentBannerIndex] = useState(0);
  const [isLoading, setIsLoading] = useState(true);

  useEffect(() => {
    const fetchBanners = async () => {
      try {
        setIsLoading(true);
        const activeBanners = await getActiveBanners('main');
        setBanners(activeBanners);
      } catch (error) {
        console.error('Error fetching banners:', error);
      } finally {
        setIsLoading(false);
      }
    };

    fetchBanners();
  }, []);

  const nextBanner = () => {
    setCurrentBannerIndex((prevIndex) => 
      prevIndex === banners.length - 1 ? 0 : prevIndex + 1
    );
  };

  const prevBanner = () => {
    setCurrentBannerIndex((prevIndex) =>
      prevIndex === 0 ? banners.length - 1 : prevIndex - 1
    );
  };

  // If no banners and not loading, show default content
  if (!isLoading && banners.length === 0) {
    return <DefaultHeroContent />;
  }

  // If still loading, show loading state
  if (isLoading) {
    return (
        <div className="my-8 mx-auto max-w-7xl p-8">
            <div className="h-[400px] bg-gray-200 animate-pulse rounded-2xl"></div>
        </div>
    );
  }

  const currentBanner = banners[currentBannerIndex];

  return (
    <div 
      className="relative bg-cover bg-center bg-no-repeat rounded-2xl overflow-hidden my-8 mx-auto max-w-7xl p-8 transition-all duration-500"
      style={{
        backgroundImage: `linear-gradient(rgba(0, 0, 0, 0.2), rgba(0, 0, 0, 0.2)), url(${formatBannerImageUrl(currentBanner.image)})`,
        minHeight: '460px',
      }}
    >
        {/* Banner Navigation Arrows */}
        {banners.length > 1 && (
          <>
            <button 
              onClick={prevBanner}
              className="absolute left-4 top-1/2 -translate-y-1/2 bg-white/80 hover:bg-white text-gray-800 p-2 rounded-full shadow-md z-10"
              aria-label="Previous banner"
            >
              <ChevronLeft className="w-6 h-6" />
            </button>
            <button 
              onClick={nextBanner}
              className="absolute right-4 top-1/2 -translate-y-1/2 bg-white/80 hover:bg-white text-gray-800 p-2 rounded-full shadow-md z-10"
              aria-label="Next banner"
            >
              <ChevronRight className="w-6 h-6" />
            </button>
          </>
        )}

      <div className="container mx-auto px-6 py-16 text-center text-white relative z-10">
        <h1 className="text-4xl md:text-5xl font-bold leading-tight text-white">
          {currentBanner.title}
        </h1>
        <p className="mt-4 text-base md:text-lg text-gray-200 max-w-2xl mx-auto">
          {currentBanner.subtitle  || 'Order medicines with prescription, consult doctors online - all from the comfort of your home'}
        </p>
        
        <div className="mt-8 max-w-xl mx-auto">
          <div className="flex items-center bg-white rounded-full p-1.5 shadow-md">
            <Search className="text-gray-400 mx-3 h-5 w-5" />
            <input
              type="text"
              placeholder="Search your needs"
              className="w-full bg-transparent focus:outline-none text-gray-700 placeholder-gray-500"
            />
            <button className="bg-blue-600 hover:bg-blue-700 text-white font-semibold py-2.5 px-8 rounded-full text-sm">
              Search
            </button>
          </div>
        </div>

        <div className="my-4 text-gray-200 text-sm">or</div>

        <button
          onClick={() => navigate('/home/upload-prescription')}
          className="bg-white text-blue-600 font-bold py-3 px-6 rounded-full shadow-md hover:bg-gray-100 flex items-center mx-auto border border-gray-200"
        >
          <Upload className="mr-2 h-5 w-5" />
          <span className="text-sm">Upload Prescription</span>
        </button>

        <div className="mt-8 flex flex-wrap justify-center items-center gap-x-8 gap-y-4 text-white">
          <div className="flex items-center text-sm font-medium">
            <ShieldCheck className="mr-2 h-5 w-5" />
            <span>100% Authentic</span>
          </div>
          <div className="flex items-center text-sm font-medium">
            <Clock className="mr-2 h-5 w-5" />
            <span>2 Hour Delivery</span>
          </div>
          <div className="flex items-center text-sm font-medium">
            <Star className="mr-2 h-5 w-5" />
            <span>4.8 Rating</span>
          </div>
        </div>
      </div>
    </div>
  );
}
