import { ChevronLeft, ChevronRight } from 'lucide-react';
import { useRef } from 'react';

interface Category {
  _id: string;
  name: string;
  image?: string;
  slug?: string;
}

interface CategorySectionProps {
  categories: Category[];
  loading?: boolean;
  onCategoryClick?: (categoryId: string) => void;
}

export default function CategorySection({ 
  categories = [], 
  loading = false,
  onCategoryClick 
}: CategorySectionProps) {
  const scrollContainerRef = useRef<HTMLDivElement>(null);

  const scroll = (direction: 'left' | 'right') => {
    if (!scrollContainerRef.current) return;
    
    const container = scrollContainerRef.current;
    const firstChild = container.firstElementChild as HTMLElement | null;
    const cardWidth = firstChild?.offsetWidth || 200; // Get width of first category card including padding and border
    const gap = 16; // Gap between cards (4 * 4px = 16px from gap-4)
    const scrollAmount = cardWidth + gap; // Scroll by one card width + gap
    
    if (direction === 'left') {
      container.scrollBy({ left: -scrollAmount, behavior: 'smooth' });
    } else {
      container.scrollBy({ left: scrollAmount, behavior: 'smooth' });
    }
  };
  return (
    <section className="py-8 sm:py-12 bg-white">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <h2 className="text-xl sm:text-2xl lg:text-3xl font-bold text-gray-900 mb-6 sm:mb-8">Shop by Category</h2>

        <div className="relative">
          <button 
            onClick={() => scroll('left')}
            className="hidden md:flex absolute left-0 top-1/2 -translate-y-1/2 -translate-x-4 w-10 h-10 bg-white rounded-full shadow-lg items-center justify-center z-10 hover:bg-gray-50 focus:outline-none"
            aria-label="Scroll left"
          >
            <ChevronLeft className="w-5 h-5 text-gray-600" />
          </button>

          <div 
            ref={scrollContainerRef}
            className="flex flex-nowrap gap-4 sm:gap-6 overflow-x-auto pb-4 scrollbar-hide snap-x snap-mandatory"
            style={{
              scrollbarWidth: 'none', // Firefox
              msOverflowStyle: 'none', // IE and Edge
            }}
          >
            {loading ? (
              // Skeleton loaders while loading
              Array(5).fill(0).map((_, index) => (
                <div key={index} className="bg-gray-50 rounded-lg sm:rounded-xl p-3 sm:p-4 lg:p-6 text-center">
                  <div className="w-20 h-20 sm:w-24 sm:h-24 lg:w-32 lg:h-32 mx-auto mb-2 sm:mb-3 lg:mb-4 bg-gray-200 rounded-lg animate-pulse"></div>
                  <div className="h-4 bg-gray-200 rounded w-3/4 mx-auto"></div>
                </div>
              ))
            ) : categories.length > 0 ? (
              // Render actual categories
              categories.map((category) => (
                <div 
                  key={category._id}
                  className="flex-shrink-0 w-[calc(50%-0.75rem)] sm:w-[calc(33.333%-1rem)] lg:w-[calc(20%-1rem)] snap-start"
                >
                  <a
                    href={`/products?category=${category._id}`}
                    onClick={(e) => {
                      e.preventDefault();
                      if (onCategoryClick) {
                        onCategoryClick(category._id);
                      } else {
                        window.location.href = `/products?category=${category._id}`;
                      }
                    }}
                    className="block bg-gray-50 rounded-lg sm:rounded-xl p-3 sm:p-4 lg:p-6 text-center hover:shadow-md transition-shadow cursor-pointer h-full"
                  >
                  <div className="w-20 h-20 sm:w-24 sm:h-24 lg:w-32 lg:h-32 mx-auto mb-2 sm:mb-3 lg:mb-4 bg-white rounded-lg flex items-center justify-center overflow-hidden">
                    {category.image ? (
                      <img 
                        src={category.image} 
                        alt={category.name}
                        className="w-full h-full object-cover"
                        onError={(e) => {
                          // Fallback to default icon if image fails to load
                          const target = e.target as HTMLImageElement;
                          target.style.display = 'none';
                          const svg = target.nextElementSibling as SVGSVGElement;
                          if (svg) svg.style.display = 'block';
                        }}
                      />
                    ) : null}
                    <svg 
                      viewBox="0 0 120 120" 
                      className={`w-16 h-16 sm:w-20 sm:h-20 lg:w-24 lg:h-24 ${category.image ? 'hidden' : ''}`}
                      style={{ display: category.image ? 'none' : 'block' }}
                    >
                      <ellipse cx="60" cy="90" rx="35" ry="8" fill="#d1d5db"/>
                      <rect x="40" y="50" width="40" height="50" fill="#e5e7eb" rx="5"/>
                      <rect x="43" y="53" width="34" height="30" fill="#f3f4f6" rx="3"/>
                      <circle cx="60" cy="35" r="10" fill="#9ca3af"/>
                      <line x1="60" y1="35" x2="60" y2="50" stroke="#6b7280" strokeWidth="2"/>
                      <rect x="50" y="63" width="20" height="1.5" fill="#3b82f6"/>
                      <rect x="50" y="70" width="20" height="1.5" fill="#3b82f6"/>
                      <rect x="50" y="77" width="20" height="1.5" fill="#3b82f6"/>
                      <circle cx="68" cy="90" r="5" fill="#6b7280"/>
                    </svg>
                  </div>
                    <h3 className="font-semibold text-gray-900 text-xs sm:text-sm lg:text-base">
                      {category.name}
                    </h3>
                  </a>
                </div>
              ))
            ) : (
              // No categories found
              <div className="w-full text-center py-8 text-gray-500">
                No categories found
              </div>
            )}
          </div>

          <button 
            onClick={() => scroll('right')}
            className="hidden md:flex absolute right-0 top-1/2 -translate-y-1/2 translate-x-4 w-10 h-10 bg-white rounded-full shadow-lg items-center justify-center z-10 hover:bg-gray-50 focus:outline-none"
            aria-label="Scroll right"
          >
            <ChevronRight className="w-5 h-5 text-gray-600" />
          </button>
        </div>
      </div>
    </section>
  );
}
