import React, { useState, useEffect, useCallback } from 'react';
import { Minus, Plus, Trash2, Check, Plus as PlusIcon, ChevronRight, Truck, Tag, X } from 'lucide-react';
import { useNavigate, Link } from 'react-router-dom';
import { useCart } from '../../context/CartContext';
import { useAuth } from '../../context/AuthContext';
import axios from 'axios';
import { toast } from 'react-hot-toast';
import Header from './Header';
import Footer from './Footer';
import AddressModal from './AddressModal';

// Define Address interface
interface Address {
  id?: string;
  _id?: string;
  name: string;
  contactNumber: string;
  address: string;
  city: string;
  state: string;
  postcode: string;
  country: string;
  landmark: string;
  isDefault: boolean;
  is_default?: boolean;
  type?: string;
  status?: boolean;
  location?: {
    type: string;
    coordinates: number[];
  };
  user?: string;
  createdAt?: string;
  updatedAt?: string;
  __v?: number;
}

// Define CartItem interface
interface CartItem {
  id: string;
  _id?: string;
  name: string;
  price: number;
  quantity: number;
  image?: string;
  brand?: string;
  category?: string;
  storeId?: string;
  storeName?: string;
  store?: string;
  discount?: number | { _id: string; name: string; type: 'percentage' | 'fixed'; value: number };
  originalPrice?: number;
  taxAmount?: number;
  discountAmount?: number;
  unit?: string;
  weight?: number;
  variantId?: string | null;
}

const CartPage: React.FC = () => {
  const navigate = useNavigate();
  const { state: cartState, removeFromCart, updateQuantity, clearCart } = useCart();
  const { user } = useAuth();
  const [savedAddresses, setSavedAddresses] = useState<Address[]>([]);
  const [selectedAddressId, setSelectedAddressId] = useState<string | null>(null);
  const [isLoading, setIsLoading] = useState<boolean>(false);
  const [error, setError] = useState<string | null>(null);
  const [isAddressModalOpen, setIsAddressModalOpen] = useState<boolean>(false);

  // Payment method state
  const [active, setActive] = useState<string>('');

  // Handle address selection
  const handleAddressSelect = (addressId: string) => {
    setSelectedAddressId(addressId);
  };

  const handleAddressAdded = (newAddress: Address) => {
    setSavedAddresses(prev => [...prev, newAddress]);
    setSelectedAddressId(newAddress.id || newAddress._id || null);
    setIsAddressModalOpen(false);
  };

  // Handle opening the address modal
  const handleAddAddress = () => {
    setIsAddressModalOpen(true);
  };
  
  // Coupon related states
  const [couponCode, setCouponCode] = useState('');
  const [appliedCoupon, setAppliedCoupon] = useState<{
    code: string;
    discount: number;
    type: 'percentage' | 'fixed';
    minCartValue?: number;
    maxDiscount?: number;
  } | null>(null);
  const [isApplyingCoupon, setIsApplyingCoupon] = useState(false);
  const [couponError, setCouponError] = useState<string | null>(null);
  
  // Fees and amounts
  const [cashHandlingFee] = useState(9);
  const [deliveryFee] = useState(0);
  const [subTotal, setSubTotal] = useState(cartState?.totalPrice || 0);
  const [discount, setDiscount] = useState(0);

  // Calculate total with coupon discount
  const calculateTotal = useCallback(() => {
    let total = subTotal + deliveryFee;
    if (active === 'cod') {
      total += cashHandlingFee;
    }
    return Math.max(0, total - discount);
  }, [subTotal, deliveryFee, cashHandlingFee, discount, active]);

  // Format currency
  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-IN', {
      style: 'currency',
      currency: 'INR',
      minimumFractionDigits: 2,
      maximumFractionDigits: 2
    }).format(amount).replace('₹', '₹');
  };

  // Check if coupon is applicable to cart
  const isCouponApplicable = useCallback((coupon: any) => {
    if (coupon.minCartValue && subTotal < coupon.minCartValue) {
      return {
        valid: false,
        message: `Minimum cart value of ₹${coupon.minCartValue} required for this coupon`
      };
    }
    return { valid: true };
  }, [subTotal]);

  // Fetch saved addresses
  const fetchAddresses = useCallback(async () => {
    const token = localStorage.getItem('token');
    if (!token) {
      setError('Please log in to view saved addresses');
      return;
    }
    try {
      setIsLoading(true);
      const response = await axios.get(`http://localhost:5000/api/v1/addresses`, {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        withCredentials: true
      });

      let addresses: Address[] = [];
      if (Array.isArray(response.data)) {
        addresses = response.data;
      } else if (response.data && Array.isArray(response.data.data)) {
        addresses = response.data.data;
      } else if (response.data?.addresses) {
        addresses = response.data.addresses;
      }

      setSavedAddresses(addresses);
      // Select default or first address
      const defaultAddress = addresses.find(addr => addr.is_default || addr.isDefault) || addresses[0];
      const addressId = defaultAddress?._id || defaultAddress?.id;
      if (addressId) setSelectedAddressId(addressId);
    } catch (err: any) {
      setError('Failed to load saved addresses. Please try again later.');
      toast.error('Failed to load addresses');
    } finally {
      setIsLoading(false);
    }
  }, []);

  useEffect(() => {
    if (user) fetchAddresses();
  }, [user, fetchAddresses]);

  useEffect(() => {
    setSubTotal(cartState?.totalPrice || 0);
  }, [cartState?.totalPrice]);

  // Apply coupon code
  const applyCoupon = useCallback(async () => {
    if (!couponCode.trim()) {
      setCouponError('Please enter a coupon code');
      return;
    }
    
    setCouponError(null);
    setIsApplyingCoupon(true);
    
    try {
      // First, fetch all active coupons
      const response = await axios.get('http://localhost:5000/api/v1/coupons/active', {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        }
      });

      if (!response.data || !response.data.success) {
        setCouponError('Failed to validate coupon. Please try again.');
        return;
      }

      const activeCoupons = response.data.data || [];
      const coupon = activeCoupons.find(
        (c: any) => c.code.toLowerCase() === couponCode.trim().toLowerCase()
      );

      if (!coupon) {
        setCouponError('Invalid or expired coupon code');
        return;
      }

      // Check if coupon is applicable
      if (coupon.minOrderAmount && subTotal < coupon.minOrderAmount) {
        setCouponError(`Minimum order amount of ₹${coupon.minOrderAmount} required for this coupon`);
        return;
      }

      // Check if coupon is expired
      const now = new Date();
      if (coupon.expiryDate && new Date(coupon.expiryDate) < now) {
        setCouponError('This coupon has expired');
        return;
      }

      // Check if coupon has started
      if (coupon.startDate && new Date(coupon.startDate) > now) {
        setCouponError('This coupon is not yet valid');
        return;
      }

      // Calculate discount amount
      let discountAmount = 0;
      if (coupon.discountType === 'percentage') {
        discountAmount = (subTotal * coupon.discountValue) / 100;
        // Apply max discount limit if specified
        if (coupon.maxDiscountAmount && discountAmount > coupon.maxDiscountAmount) {
          discountAmount = coupon.maxDiscountAmount;
        }
      } else {
        discountAmount = coupon.discountValue;
      }

      // Ensure discount doesn't exceed cart total
      if (discountAmount > subTotal) {
        discountAmount = subTotal;
      }

      // If we get here, the coupon is valid
      const couponData = {
        code: coupon.code,
        discount: coupon.discountValue,
        type: coupon.discountType,
        minCartValue: coupon.minOrderAmount,
        maxDiscount: coupon.maxDiscountAmount,
        expiryDate: coupon.expiryDate,
        description: coupon.description
      };

      // Save coupon data to localStorage
      localStorage.setItem('appliedCoupon', JSON.stringify({
        code: coupon.code,
        discount: coupon.discountValue,
        type: coupon.discountType,
        minCartValue: coupon.minOrderAmount,
        maxDiscount: coupon.maxDiscountAmount,
        appliedAt: new Date().toISOString()
      }));
      
      setAppliedCoupon(couponData);
      setDiscount(discountAmount);
      setCouponError(null);
      toast.success('Coupon applied successfully!');
    } catch (error: any) {
      const errorMessage = error.response?.data?.message || 
        error.response?.data?.error || 
        error.message || 
        'Failed to apply coupon. Please try again.';
      setCouponError(errorMessage);
      
      if (error.response?.status === 401) {
        navigate('/login');
      }
    } finally {
      setIsApplyingCoupon(false);
    }
  }, [couponCode, subTotal, navigate]);

  const removeCoupon = useCallback(() => {
    // Remove coupon from localStorage
    localStorage.removeItem('appliedCoupon');
    
    setAppliedCoupon(null);
    setCouponCode('');
    setDiscount(0);
    setCouponError(null);
    toast.success('Coupon removed successfully');
  }, []);

  // Proceed to payment and checkout
  const handleProceedToCheckout = async () => {
    console.log('Proceeding to checkout...');
    
    // Check if user is authenticated
    if (!user) {
      console.log('User not authenticated, redirecting to login...');
      toast.error('Please log in to proceed to checkout');
      navigate('/login', { 
        state: { from: '/cart' },
        replace: true 
      });
      return;
    }

    // Check if cart is empty
    if (cartState.items.length === 0) {
      console.log('Cart is empty');
      toast.error('Your cart is empty');
      return;
    }

    // Check if address is selected
    if (!selectedAddressId) {
      console.log('No address selected');
      toast.error('Please select a delivery address');
      return;
    }
    
    // Find the selected address
    const selectedAddress = savedAddresses.find(
      addr => (addr.id === selectedAddressId || addr._id === selectedAddressId)
    );
    
    if (!selectedAddress) {
      console.log('Selected address not found');
      toast.error('Selected address not found');
      return;
    }
    
    // Group cart items by store
    const itemsByStore: { [storeId: string]: any[] } = {};
    cartState.items.forEach(item => {
      const storeId = (item as any).storeId || 'default';
      if (!itemsByStore[storeId]) {
        itemsByStore[storeId] = [];
      }
      itemsByStore[storeId].push(item);
    });
    
    // Prepare checkout data with store information
    const checkoutData = {
      // Shipping information
      shippingAddress: {
        fullName: selectedAddress.name,
        contactNumber: selectedAddress.contactNumber || '',
        street: selectedAddress.address,
        city: selectedAddress.city,
        state: selectedAddress.state,
        zipCode: selectedAddress.postcode,
        country: selectedAddress.country || 'India',
        landmark: selectedAddress.landmark || '',
        isDefault: selectedAddress.isDefault || false
      },
      
      // Order items with store information
      orderItems: cartState.items.map(item => ({
        productId: item.id,
        _id: item._id,
        name: item.name,
        price: item.price,
        originalPrice: item.originalPrice || item.price,
        quantity: item.quantity,
        image: item.image,
        storeId: (item as any).storeId || null,
        storeName: (item as any).storeName || null,
        brand: item.brand,
        category: item.category,
        unit: item.unit,
        taxAmount: item.taxAmount || 0,
        discountAmount: item.discountAmount || 0,
        discount: item.discount || null
      })),
      
      // Order summary
      orderSummary: {
        subtotal: cartState.items.reduce((acc, item) => acc + ((item.originalPrice || item.price) * item.quantity), 0),
        delivery: deliveryFee,
        discount: discount,
        tax: cartState.items.reduce((acc, item) => acc + (item.taxAmount || 0) * item.quantity, 0),
        total: calculateTotal(),
        itemsCount: cartState.totalItems,
        storesCount: Object.keys(itemsByStore).length
      },
      
      // Payment information
      payment: {
        method: active === 'cod' ? 'cash_on_delivery' : 'online_payment',
        status: 'pending',
        cashHandlingFee: active === 'cod' ? cashHandlingFee : 0
      },
      
      // Coupon information if applied
      coupon: appliedCoupon ? {
        code: appliedCoupon.code,
        discount: discount,
        type: appliedCoupon.type,
        minCartValue: appliedCoupon.minCartValue,
        maxDiscount: appliedCoupon.maxDiscount
      } : null,
      
      // Store information
      stores: Object.keys(itemsByStore).map(storeId => {
        const storeItems = itemsByStore[storeId];
        const storeName = storeItems[0]?.storeName || 'Default Store';
        const storeSubtotal = storeItems.reduce((sum, item) => sum + (item.price * item.quantity), 0);
        
        return {
          storeId: storeId === 'default' ? null : storeId,
          storeName,
          itemCount: storeItems.reduce((sum, item) => sum + item.quantity, 0),
          subtotal: storeSubtotal
        };
      })
    };
    
    // Save to localStorage for persistence
    localStorage.setItem('checkoutData', JSON.stringify(checkoutData));
    
    console.log('Navigating to checkout with data:', checkoutData);
    
    try {
      // Navigate to checkout process with the complete order data
      navigate('/checkout', { 
        state: { 
          from: 'cart',
          ...checkoutData 
        },
        replace: true 
      });
    } catch (error) {
      console.error('Navigation error:', error);
      toast.error('Failed to proceed to checkout. Please try again.');
    }
  };

  // Empty cart view
  if (cartState.items.length === 0) {
    return (
      <div className="min-h-screen bg-white flex flex-col">
        <Header />
        <div className="flex-1 flex items-center justify-center p-4">
          <div className="text-center">
            <p className="text-xl text-gray-600 mb-4">Your cart is empty</p>
            <button
              onClick={() => navigate('/products')}
              className="bg-blue-600 text-white px-6 py-2 rounded-md hover:bg-blue-700 transition-colors"
            >
              Continue Shopping
            </button>
          </div>
        </div>
        <Footer />
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50 flex flex-col">
      <Header />
      {/* Breadcrumb */}
      <div className="bg-gray-50 py-3 px-4">
        <div className="max-w-6xl mx-auto">
          <nav className="flex items-center space-x-2 text-sm">
            <Link to="/" className="text-gray-600 hover:text-blue-600 transition-colors">
              Home
            </Link>
            <ChevronRight className="w-4 h-4 text-gray-400" />
            <span className=" text-blue-800 font-medium">Cart</span>
          </nav>
        </div>
      </div>
      <div className="flex-1 py-6 px-4">
        <div className="max-w-6xl mx-auto w-full flex flex-col lg:flex-row gap-6">
          {/* LEFT SECTION */}
          <div className="flex-1">
            <h2 className="text-xs font-semibold text-blue-800 mb-2">MY CART</h2>
            {/* Address Box */}
            <div className="border rounded-lg px-4 py-3 bg-white mb-6">
              {savedAddresses.length > 0 ? (
                <div className="flex justify-between items-start">
                  <div>
                    <p className="text-sm text-gray-500">
                      <span className="font-semibold text-gray-800">Bill to</span> {savedAddresses[0].name}
                    </p>
                    <p className="text-sm text-gray-700 mt-1">
                      {savedAddresses[0].address}{savedAddresses[0].landmark ? `, ${savedAddresses[0].landmark}` : ''}, {savedAddresses[0].city}, {savedAddresses[0].state} - {savedAddresses[0].postcode}
                    </p>
                  </div>
                  <button 
                    onClick={handleAddAddress}
                    className="text-orange-500 font-semibold text-sm whitespace-nowrap ml-4"
                  >
                    CHANGE
                  </button>
                </div>
              ) : (
                <div className="flex justify-between items-center">
                  <p className="text-sm text-gray-500">
                    <span className="font-semibold text-gray-800">Delivery Address</span>
                  </p>
                  <button 
                    onClick={handleAddAddress}
                    className="flex items-center text-orange-500 font-semibold text-sm py-1 px-3 border border-orange-500 rounded hover:bg-orange-50 transition-colors"
                  >
                    <PlusIcon size={14} className="mr-1" />
                    ADD ADDRESS
                  </button>
                </div>
              )}
            </div>
            {/* Cart Header */}
            <div className="flex justify-between items-center mb-3">
              <h3 className="text-xs font-semibold text-blue-800">
                {cartState.items.length} ITEM{cartState.items.length !== 1 ? 'S' : ''} IN YOUR CART
              </h3>
              <button 
                onClick={() => navigate('/products')}
                className="text-orange-500 font-semibold text-sm"
              >
                ADD ITEMS
              </button>
            </div>
            {/* Product Card */}
            {cartState.items.length > 0 ? (
              <div className="border rounded-lg bg-white p-4">
                {/* Top Row */}
                <div className="flex justify-between items-center mb-3">
                  <div className="flex items-center gap-2">
                    <Truck className="w-4 h-4 text-blue-600" />
                    <p className="font-medium text-gray-800">Delivery in 30 mins</p>
                  </div>
                  <p className="text-xs text-gray-500">Shipment 1/1</p>
                </div>
                <hr className="mb-4" />
                {/* Product List */}
                {cartState.items.map((item) => (
                  <div key={item.id} className="flex gap-4 mb-4 last:mb-0">
                    <Link to={`/product/${item.id}`} className="flex-shrink-0">
                      <img
                        src={item.image || 'https://via.placeholder.com/64x80'}
                        alt={item.name}
                        className="w-16 h-20 object-contain rounded-md border hover:opacity-90 transition-opacity"
                      />
                    </Link>
                    <div className="flex-1">
                      <Link to={`/product/${item.id}`} className="hover:underline">
                        <h4 className="text-sm font-medium leading-snug text-gray-800 hover:text-blue-600 transition-colors">
                          {item.name}
                        </h4>
                      </Link>
                      <p className="text-xs text-gray-500 mt-1">
                        {item.unit || '1 unit'}
                        {(item.storeName || item.store) && (
                          <span className="ml-2 text-gray-600 font-medium">
                            • {item.storeName || item.store}
                          </span>
                        )}
                      </p>
                      <div className="mt-2 flex items-center gap-2">
                        {item.originalPrice && item.originalPrice > item.price && (
                          <p className="text-gray-400 line-through text-sm">
                            ₹{item.originalPrice.toFixed(2)}
                          </p>
                        )}
                        <p className="text-gray-800 font-semibold text-sm">
                          ₹{item.price.toFixed(2)}
                        </p>
                      </div>
                      {/* Quantity Controls */}
                      <div className="mt-3 flex items-center gap-3">
                        <div className="flex items-center border rounded-full px-2">
                          <button 
                            onClick={() => updateQuantity(item.id, Math.max(1, item.quantity - 1))}
                            className="text-lg px-2"
                          >−</button>
                          <span className="text-sm font-medium px-1">{item.quantity}x</span>
                          <button 
                            onClick={() => updateQuantity(item.id, item.quantity + 1)}
                            className="text-lg px-2"
                          >＋</button>
                        </div>
                        <button 
                          onClick={() => removeFromCart(item.id)}
                          className="text-sm text-gray-500 hover:text-red-500 transition"
                        >Remove</button>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            ) : (
              <div className="border rounded-lg bg-white p-8 text-center">
                <p className="text-gray-500 mb-4">Your cart is empty</p>
                <button
                  onClick={() => navigate('/products')}
                  className="px-4 py-2 bg-blue-600 text-white rounded-lg text-sm font-medium hover:bg-blue-700"
                >
                  Continue Shopping
                </button>
              </div>
            )}
          </div>
          {/* RIGHT SECTION */}
          <div className="w-full lg:w-80 h-fit sticky top-6">
            <div className="bg-white rounded-lg shadow p-6">
              <h2 className="text-lg font-semibold mb-4">Order Summary</h2>
              <div className="space-y-4">
                {/* Items List */}
                <div className="space-y-3 mb-4 max-h-60 overflow-y-auto pr-2">
                  {cartState.items.map((item) => (
                    <div key={item.id} className="flex items-center justify-between">
                      <div className="flex items-center gap-3">
                        <div className="w-12 h-12 bg-gray-100 rounded-md overflow-hidden">
                          {item.image ? (
                            <img 
                              src={item.image} 
                              alt={item.name} 
                              className="w-full h-full object-cover"
                            />
                          ) : (
                            <div className="w-full h-full flex items-center justify-center bg-gray-200 text-gray-400">
                              <Truck className="w-5 h-5" />
                            </div>
                          )}
                        </div>
                        <div>
                          <div className="text-sm font-medium text-gray-800">{item.name}</div>
                          <div className="text-xs text-gray-500">Qty: {item.quantity}</div>
                        </div>
                      </div>
                      <div className="text-sm font-medium">
                        ₹{(item.price * item.quantity).toFixed(2)}
                      </div>
                    </div>
                  ))}
                </div>
                {/* Coupon Code Section */}
                <div className="border-t border-gray-200 pt-4 mt-4">
                  <h3 className="text-sm font-medium text-gray-700 mb-3">Have a coupon code?</h3>
                  
                  {appliedCoupon ? (
                    <div className="mb-4">
                      <div className="flex justify-between items-center bg-green-50 border border-green-100 rounded-lg p-3">
                        <div className="flex items-center gap-2">
                          <Tag className="w-4 h-4 text-green-600" />
                          <div>
                            <span className="text-sm font-medium text-green-800">
                              {appliedCoupon.code} Applied
                            </span>
                            {appliedCoupon.type === 'percentage' ? (
                              <p className="text-xs text-green-600">
                                {appliedCoupon.discount}% off (Max. {formatCurrency(appliedCoupon.maxDiscount || Infinity)})
                              </p>
                            ) : (
                              <p className="text-xs text-green-600">
                                {formatCurrency(appliedCoupon.discount)} off
                              </p>
                            )}
                          </div>
                        </div>
                        <button 
                          onClick={removeCoupon}
                          className="text-green-600 hover:text-green-800 p-1 -mr-2"
                          aria-label="Remove coupon"
                        >
                          <X className="w-4 h-4" />
                        </button>
                      </div>
                    </div>
                  ) : (
                    <div className="space-y-2">
                      <div className="flex gap-2">
                        <div className="flex-1 relative">
                          <input
                            type="text"
                            placeholder="Enter coupon code"
                            value={couponCode}
                            onChange={(e) => {
                              setCouponCode(e.target.value);
                              setCouponError(null);
                            }}
                            onKeyDown={(e) => e.key === 'Enter' && applyCoupon()}
                            className="w-full border border-gray-300 rounded-lg px-4 py-2 text-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                            aria-label="Coupon code"
                          />
                        </div>
                        <button
                          onClick={applyCoupon}
                          disabled={!couponCode.trim() || isApplyingCoupon}
                          className={`px-4 py-2 rounded-lg text-sm font-medium whitespace-nowrap ${
                            couponCode.trim() && !isApplyingCoupon
                              ? 'bg-blue-600 text-white hover:bg-blue-700'
                              : 'bg-gray-100 text-gray-400 cursor-not-allowed'
                          } transition-colors`}
                        >
                          {isApplyingCoupon ? 'Applying...' : 'Apply'}
                        </button>
                      </div>
                      {couponError && (
                        <p className="text-red-500 text-xs mt-1">{couponError}</p>
                      )}
                      {!appliedCoupon && subTotal > 0 && (
                        <p className="text-xs text-gray-500">
                          Apply a coupon code at checkout to get a discount
                        </p>
                      )}
                    </div>
                  )}
                </div>

                {/* Order Summary */}
                <div className="space-y-3 mt-6">
                  <div className="flex justify-between items-center text-gray-700 text-sm">
                    <span>Subtotal ({cartState.totalItems} {cartState.totalItems === 1 ? 'item' : 'items'})</span>
                    <span>{formatCurrency(subTotal)}</span>
                  </div>
                  
                  {discount > 0 && (
                    <div className="flex justify-between items-center text-green-600 text-sm">
                      <span>Coupon Discount</span>
                      <span>-{formatCurrency(discount)}</span>
                    </div>
                  )}
                  
                  {deliveryFee > 0 ? (
                    <div className="flex justify-between items-center text-gray-700 text-sm">
                      <span>Delivery Fee</span>
                      <span>{formatCurrency(deliveryFee)}</span>
                    </div>
                  ) : (
                    <div className="flex justify-between items-center text-green-600 text-sm">
                      <span>Delivery</span>
                      <span>Free</span>
                    </div>
                  )}
                  
                  {active === 'cod' && cashHandlingFee > 0 && (
                    <div className="flex justify-between items-center text-gray-700 text-sm">
                      <span>Cash Handling Fee</span>
                      <span>{formatCurrency(cashHandlingFee)}</span>
                    </div>
                  )}
                  
                  <div className="border-t border-gray-200 pt-3 mt-2">
                    <div className="flex justify-between font-semibold">
                      <span>Total Amount</span>
                      <span>{formatCurrency(calculateTotal())}</span>
                    </div>
                    {discount > 0 && (
                      <p className="text-xs text-green-600 mt-1 text-right">
                        You saved {formatCurrency(discount)} with this order
                      </p>
                    )}
                  </div>
                </div>
                <button
                  onClick={handleProceedToCheckout}
                  className="w-full bg-orange-500 text-white py-2 rounded-md hover:bg-orange-600 transition-colors"
                  disabled={savedAddresses.length === 0}
                >
                  PROCEED TO CHECKOUT
                </button>
                {savedAddresses.length === 0 && (
                  <p className="text-sm text-red-500 mt-2">Please add a delivery address to continue</p>
                )}
              </div>
            </div>
          </div>
        </div>
      </div>
      <Footer />
      {/* Address Modal */}
      {isAddressModalOpen && (
        <AddressModal
          isOpen={isAddressModalOpen}
          onClose={() => setIsAddressModalOpen(false)}
          onAddressAdded={handleAddressAdded}
        />
      )}
    </div>
  );
};

export default CartPage;
