const express = require('express');
const router = express.Router();
const addressesController = require('../controllers/addressesController');
const { auth } = require('../middleware/auth');
const { 
  validateCreateAddress, 
  validateUpdateAddress 
} = require('../validations/address.validation');

// @desc    Get all addresses for current user
// @route   GET /api/addresses
// @access  Private
router.get('/', auth, addressesController.getUserAddresses);

// @desc    Create or update address for current user
// @route   POST /api/addresses
// @access  Private
router.post('/', auth, validateCreateAddress, addressesController.createOrUpdateAddress);

// @desc    Set default address for current user
// @route   PUT /api/addresses/:id/default
// @access  Private
router.put('/:id/default', auth, addressesController.setDefaultAddress);

// @desc    Get single address by ID
// @route   GET /api/addresses/:id
// @access  Private
router.get('/:id', auth, addressesController.getAddress);

// @desc    Update an address
// @route   PUT /api/addresses/:id
// @access  Private
router.put('/:id', auth, validateUpdateAddress, addressesController.updateAddress);

// @desc    Delete an address (Admin only)
// @route   DELETE /api/addresses/:id
// @access  Private/Admin
router.delete('/:id', auth, addressesController.deleteAddress);

module.exports = router;
